unit mCoverSheetDisplayPanel_CPRS_Reminders;
{
  ================================================================================
  *
  *       Application:  CPRS - Coversheet
  *       Developer:    dan.petit@domain.ext
  *       Site:         Salt Lake City ISC
  *       Date:         2015-12-08
  *
  *       Description:  Customized display panel for Clinical Reminders.
  *
  *       Notes:
  *
  ================================================================================
}

interface

uses
  Winapi.Windows,
  Winapi.Messages,
  System.SysUtils,
  System.Variants,
  System.Classes,
  Vcl.Graphics,
  Vcl.Controls,
  Vcl.Forms,
  Vcl.Dialogs,
  Vcl.ExtCtrls,
  Vcl.Menus,
  Vcl.ImgList,
  Vcl.ComCtrls,
  Vcl.StdCtrls,
  Vcl.Buttons,
  iCoverSheetIntf,
  mCoverSheetDisplayPanel_CPRS,
  oDelimitedString;

type
  TfraCoverSheetDisplayPanel_CPRS_Reminders = class(TfraCoverSheetDisplayPanel_CPRS)
  private
    fSeparator: TMenuItem;
    fClinicalMaintenance: TMenuItem;
    fEducationTopicDefinition: TMenuItem;
    fReminderInquiry: TMenuItem;
    fReferenceInquiry: TMenuItem;
    fReminderIconLegend: TMenuItem;
  protected
    { Inherited methods }
    function getTitle: string; override;
    procedure setTitle(const aValue: string); override;

    { Inherited events - TfraGridPanel }
    procedure OnPopupMenu(Sender: TObject); override;
    procedure OnPopupMenuInit(Sender: TObject); override;
    procedure OnPopupMenuFree(Sender: TObject); override;

    { Inherited events - TfraCoverSheetDisplayPanel_CPRS }
    procedure OnAddItems(aList: TStrings); override;
    procedure OnGetDetail(aRec: TDelimitedString; aResult: TStrings); override;

    { Introduced methods }
    procedure GetEducationTopics(aMenuItem: TMenuItem; aReminderIEN: integer); virtual;
    procedure GetReferenceInquiries(aMenuItem: TMenuItem; aReminderIEN: integer); virtual;

    { Introduced events }
    procedure OnReminderInquiry(Sender: TObject); virtual;
    procedure OnReferenceInquiry(Sender: TObject); virtual;
    procedure OnReminderIconLegend(Sender: TObject); virtual;
    procedure OnClinicalMaintenance(Sender: TObject); virtual;
    procedure OnEducationTopic(Sender: TObject); virtual;
  public
    constructor Create(aOwner: TComponent); override;
    destructor Destroy; override;
  end;

var
  fraCoverSheetDisplayPanel_CPRS_Reminders: TfraCoverSheetDisplayPanel_CPRS_Reminders;

implementation

uses
  uCore,
  uReminders,
  rReminders,
  fIconLegend,
  ORFn,
  ORNet;

const
  CUSTOM_TITLE = 'Clinical Reminders'; // The original title has column headers in it.

{$R *.dfm}
  { TfraCoverSheetDisplayPanel_CPRS_Reminders }

constructor TfraCoverSheetDisplayPanel_CPRS_Reminders.Create(aOwner: TComponent);
begin
  inherited;

  AddColumn(0, 'Reminder');
  AddColumn(1, 'Due Date');
  CollapseColumns;
end;

destructor TfraCoverSheetDisplayPanel_CPRS_Reminders.Destroy;
begin

  inherited;
end;

procedure TfraCoverSheetDisplayPanel_CPRS_Reminders.OnGetDetail(aRec: TDelimitedString; aResult: TStrings);
var
  aList: TStringList;
begin
  aList := TStringList.Create;
  try
    DetailReminder(aRec.GetPieceAsInteger(1), aList);
    aResult.Text := aList.Text;
  finally
    FreeAndNil(aList);
  end;

end;

procedure TfraCoverSheetDisplayPanel_CPRS_Reminders.GetEducationTopics(aMenuItem: TMenuItem; aReminderIEN: integer);
var
  aSubItem: TMenuItem;
begin
  for aSubItem in aMenuItem do
    aSubItem.Free;
  aMenuItem.Clear;

  { This needs to call out to VistA and get the menu items }

  aSubItem := NewItem('Education Item A', 0, False, True, OnEducationTopic, 0, 'pmnEduItemA');
  aSubItem.Tag := 1;
  aMenuItem.Add(aSubItem);

  aSubItem := NewItem('Education Item B', 0, False, True, OnEducationTopic, 0, 'pmnEduItemB');
  aSubItem.Tag := 2;
  aMenuItem.Add(aSubItem);

  aSubItem := NewItem('Education Item C', 0, False, True, OnEducationTopic, 0, 'pmnEduItemC');
  aSubItem.Tag := 3;
  aMenuItem.Add(aSubItem);

  aMenuItem.Enabled := (aMenuItem.Count > 0);
end;

procedure TfraCoverSheetDisplayPanel_CPRS_Reminders.GetReferenceInquiries(aMenuItem: TMenuItem; aReminderIEN: integer);
var
  aSubItem: TMenuItem;
begin
  for aSubItem in aMenuItem do
    aSubItem.Free;
  aMenuItem.Clear;

  { This needs to call out to VistA and get the menu items }

  aSubItem := NewItem('Reference Item A', 0, False, True, OnEducationTopic, 0, 'pmnEduItemA');
  aSubItem.Tag := 1;
  aMenuItem.Add(aSubItem);

  aSubItem := NewItem('Reference Item B', 0, False, True, OnEducationTopic, 0, 'pmnEduItemB');
  aSubItem.Tag := 2;
  aMenuItem.Add(aSubItem);

  aSubItem := NewItem('Reference Item C', 0, False, True, OnEducationTopic, 0, 'pmnEduItemC');
  aSubItem.Tag := 3;
  aMenuItem.Add(aSubItem);

  aMenuItem.Enabled := (aMenuItem.Count > 0);
end;

procedure TfraCoverSheetDisplayPanel_CPRS_Reminders.OnAddItems(aList: TStrings);
var
  aRec: TDelimitedString;
  aStr: string;
begin
  try
    lvData.Items.BeginUpdate;
    for aStr in aList do
      begin
        aRec := TDelimitedString.Create(aStr);

        if lvData.Items.Count = 0 then
          if aRec.GetPieceIsNull(1) and (aList.Count = 1) then
            CollapseColumns
          else
            ExpandColumns;

        with lvData.Items.Add do
          begin
            Caption := aRec.GetPiece(2);
            if aRec.GetPieceIsDouble(3) then
              SubItems.Add(FormatDateTime('MMM DD, YYYY', aRec.GetPieceAsTDateTime(3)))
            else
              SubItems.Add(aRec.GetPiece(3));
            Data := aRec;
          end;
      end;
  finally
    lvData.Items.EndUpdate;
  end;
end;

function TfraCoverSheetDisplayPanel_CPRS_Reminders.getTitle: string;
begin
  Result := CUSTOM_TITLE;
end;

procedure TfraCoverSheetDisplayPanel_CPRS_Reminders.OnPopupMenu(Sender: TObject);
var
  aRec: TDelimitedString;
begin
  inherited;

  aRec := ListViewItemRec;

  fEducationTopicDefinition.Clear;
  fReferenceInquiry.Clear;

  fClinicalMaintenance.Enabled := False;
  fEducationTopicDefinition.Enabled := False;
  fReminderInquiry.Enabled := False;
  fReferenceInquiry.Enabled := False;
  fReminderIconLegend.Enabled := True; { Always enabled }

  if Assigned(aRec) then
    if aRec.GetPieceAsInteger(1) > 0 then
      begin
        fClinicalMaintenance.Enabled := True;
        fReminderInquiry.Enabled := True;
        GetEducationTopics(fEducationTopicDefinition, aRec.GetPieceAsInteger(1));
        GetReferenceInquiries(fReferenceInquiry, aRec.GetPieceAsInteger(1));
      end;
end;

procedure TfraCoverSheetDisplayPanel_CPRS_Reminders.OnPopupMenuFree(Sender: TObject);
var
  aSubMenu: TMenuItem;
begin
  { Free all subitems from fEducationTopicDefinition and the clear it }
  for aSubMenu in fEducationTopicDefinition do
    aSubMenu.Free;
  fEducationTopicDefinition.Clear;

  { Free all subitems from fReferenceInquiry and the clear it }
  for aSubMenu in fReferenceInquiry do
    aSubMenu.Free;
  fReferenceInquiry.Clear;

  FreeAndNil(fSeparator);
  FreeAndNil(fClinicalMaintenance);
  FreeAndNil(fEducationTopicDefinition);
  FreeAndNil(fReminderInquiry);
  FreeAndNil(fReferenceInquiry);
  FreeAndNil(fReminderIconLegend);

  inherited;
end;

procedure TfraCoverSheetDisplayPanel_CPRS_Reminders.OnPopupMenuInit(Sender: TObject);
begin
  inherited;

  fSeparator := NewLine;
  fClinicalMaintenance := NewItem('Clinical Maintenance ...', 0, False, False, OnClinicalMaintenance, 0, 'pmnClinicalMaintenance');
  fEducationTopicDefinition := NewItem('Education Topic ...', 0, False, False, nil, 0, 'pmnEducationTopicDefinition');
  fReminderInquiry := NewItem('Reminder Inquiry ...', 0, False, False, OnReminderInquiry, 0, 'pmnReminderInquiry');
  fReferenceInquiry := NewItem('Reference Inquiry ...', 0, False, False, OnReferenceInquiry, 0, 'pmnReferenceInquiry');
  fReminderIconLegend := NewItem('Reminder Icon Legend ...', 0, False, False, OnReminderIconLegend, 0, 'pmnReminderIconLegend');

  pmn.Items.Add(fSeparator);
  pmn.Items.Add(fClinicalMaintenance);
  pmn.Items.Add(fEducationTopicDefinition);
  pmn.Items.Add(fReminderInquiry);
  pmn.Items.Add(fReferenceInquiry);
  pmn.Items.Add(fReminderIconLegend);
end;

procedure TfraCoverSheetDisplayPanel_CPRS_Reminders.OnClinicalMaintenance(Sender: TObject);
var
  aTxt: TStringList;
begin
  if lvData.Selected.Data <> nil then
    begin
      aTxt := TStringList.Create;
      try
        OnGetDetail(TDelimitedString(lvData.Selected.Data), aTxt);
        OnShowDetail(aTxt, 'Clinical Maintenance: ' + lvData.Selected.Caption, True);
      finally
        FreeAndNil(aTxt);
      end;
    end;
end;

procedure TfraCoverSheetDisplayPanel_CPRS_Reminders.OnEducationTopic(Sender: TObject);
begin
  with Sender as TMenuItem do
    ShowMessageFmt('Caption: %s - Tag: %d', [Caption, Tag]);
end;

procedure TfraCoverSheetDisplayPanel_CPRS_Reminders.OnReferenceInquiry(Sender: TObject);
begin
  //
end;

procedure TfraCoverSheetDisplayPanel_CPRS_Reminders.OnReminderIconLegend(Sender: TObject);
begin
  ShowIconLegend(ilReminders);
end;

procedure TfraCoverSheetDisplayPanel_CPRS_Reminders.OnReminderInquiry(Sender: TObject);
var
  aRec: TDelimitedString;
  aDetail: TStringList;
begin
  aRec := ListViewItemRec;
  if aRec <> nil then
    if aRec.GetPiece(1) <> '' then
      begin
        aDetail := TStringList.Create;
        try
          OnGetDetail(aRec, aDetail);
          OnShowDetail(aDetail);
        finally
          FreeAndNil(aDetail);
        end;
      end;
end;

procedure TfraCoverSheetDisplayPanel_CPRS_Reminders.setTitle(const aValue: string);
begin
  lblTitle.Caption := CUSTOM_TITLE;
end;

end.
